//
//  CMS_Arrays.swift
//  Composr Mobile SDK
//
//  Created by Aaswini on 06/08/14.
//  Copyright (c) 2014 Aaswini. All rights reserved.
//
/*
 
 CMS_Arrays
 
 array collapse_1d_complexity(string key, array arr) - takes a list of maps and turns it into a simple list by extracting just one element from each map
 map collapse_2d_complexity(string keyKey, string valKey, array arr) - takes a list of maps and turns it into a simple map by extracting just two elements from each map
 array explode(string sep, string str) - see the PHP manual
 string implode(string sep, array arr) - see the PHP manual
 
 */

import UIKit
import Foundation

class CMS_Arrays: NSObject {
    
    /**
     *  Takes a list of maps and turns it into a simple list by extracting just one element from each map
     *
     *  @param key key for which value needs to be fetched
     *  @param arr source array containing many dictionaries
     *
     *  @return array of values corresponding to the specified key from each dictionary
     */
    class func collapse_1d_complexity(key: String, arr: [NSArray]) -> [AnyObject] {
        return (arr.map(key) as! String)
    }

    /**
     *  Takes a list of maps and turns it into a simple dictionary by extracting just two elements from each dictionary.
     *
     *  @param keyKey key for which value to be fetched. This value is kept as key in the returning dictionary.
     *  @param valKey key for which value to be fetched to be kept as value for the first parameter in returning dictionary
     *  @param arr    source array of dictionaries
     *
     *  @return dictionary with keyvalue pair for the supplied keys from all dictionaries in the source array
     */
    class func collapse_2d_complexity(keyKey: String, valKey: String, arr: [AnyObject]) -> [NSObject : AnyObject] {
        var result = [NSObject : AnyObject]()
        for dict: [NSObject : AnyObject] in arr {
            var key = (dict.valueForKey(keyKey) as! String)
            var value = (dict.valueForKey(valKey) as! String)
            result[key] = value
        }
        return result
    }

    /**
     *  Returns an array by splitting the string with the supplied seperator
     *
     *  @param sep separator
     *  @param str source string
     *
     *  @return array of components in string by splitting with separator
     */
    class func explode(sep: String, str: String) -> [AnyObject] {
        return str.componentsSeparatedByString(sep)
    }

    /**
     *  Returns a string by joining the supplied array with the supplied seperator
     *
     *  @param sep separator
     *  @param arr source array
     *
     *  @return returns string by joining the components in the array with the separator
     */
    class func implode(sep: String, arr: [AnyObject]) -> String {
        return (arr as NSArray).componentsJoinedByString(sep)
    }
    // returns values in the dictionary as list. If useKey, returns keys as list.

    /**
     *  Returns array of keys or values from a dictionary
     *
     *  @param useKey YES - returns array of keys, NO- returns array of values
     *  @param dict   source dictionary
     *
     *  @return returns array of keys or values from the dictionary
     */
    class func Array_values(useKey: Bool, dict: [NSObject : AnyObject]) -> [AnyObject] {
        var result = [AnyObject]()
        for key: String in dict.allKeys() {
            if useKey {
                result.append(key)
            }
            else {
                result.append((dict.valueForKey(key) as! String))
            }
        }
        return result
    }

    /**
     *  Returns if the array contains the given key
     *
     *  @param key   value to check for
     *  @param array source array
     *
     *  @return YES if array contains key else NO
     */
    class func in_array(key: AnyObject, array: [AnyObject]) -> Bool {
        return array.contains(key)
    }

    /**
     *  Merges two arrays to a single array
     *
     *  @param array1 source array 1
     *  @param array2 source array 2
     *
     *  @return single array containing values of array1 and array2
     */
    class func array_merge(array1: [AnyObject], array2: [AnyObject]) -> [AnyObject] {
        return array1 + array2
    }

    /**
     *  Check of given map contains the given key
     *
     *  @param key String key to check for
     *  @param map source map
     *
     *  @return YES if map contains the key else NO
     */
    class func array_key_exists(key: AnyObject, map: [NSObject : AnyObject]) -> Bool {
        return map.allKeys().contains(key)
    }

    /**
     *  Removes duplicate values from array
     *
     *  @param array source array
     *
     *  @return Returns array by removing duplicate values
     */
    class func array_unique(array: [AnyObject]) -> [AnyObject] {
        return NSOrderedSet.orderedSetWithArray(array)()
    }

    /**
     *  Sorts the given array of maps based on value of a given key
     *
     *  @param array source array of maps
     *  @param key   key to sort the maps on
     */
    class func sort_maps_by(array: [AnyObject], key: String) {
        array = (array as NSArray).sortedArrayUsingComparator({(obj1: [NSObject : AnyObject], obj2: [NSObject : AnyObject]) -> NSComparisonResult in
            if !(obj1 is obj2) {
                return "\(obj1[key] as! String)".compare("\(obj2[key] as! String)", options: .CaseInsensitiveSearch)
            }
            else if ((obj1[key] as! String) is Int) {
                return (obj1[key] as! String).compare((obj2[key] as! String))
            }

            return (obj1[key] as! String).compare((obj2[key] as! String), options: .CaseInsensitiveSearch)
        })
    }

    /**
     *  Sort array of strings or numbers
     *
     *  @param array source array of strings or numbers
     */
    class func sort(array: AnyObject) {
        array = (array as NSArray).sortedArrayUsingComparator({(obj1: AnyObject, obj2: AnyObject) -> NSComparisonResult in
            if !(obj1 is obj2) {
                return "\(obj1)".compare("\(obj2)", options: .CaseInsensitiveSearch)
            }
            else if (obj1 is Int) {
                return obj1.compare(obj2)
            }

            return obj1.compare(obj2, options: .CaseInsensitiveSearch)
        })
    }

    /**
     *  Returns number of elements in an array or map
     *
     *  @param array source array or map
     *
     *  @return number of elements in array or map
     */
    class func count(array: AnyObject) -> Int {
        if (array is [AnyObject]) {
            return Int((array as! [AnyObject]).count)
        }
        else if (array is [NSObject : AnyObject]) {
            return Int((array as! [NSObject : AnyObject]).count)
        }
        else if (array is String) {
            return Int(CMS_Strings.strlen(array))
        }

        return 0
    }

    /**
     *  Returns map corresponding to a key from array of maps
     *
     *  @param key  key
     *  @param dict array of maps
     *
     *  @return map
     */
    class func list_to_map(key: String, dict: [AnyObject]) -> [NSObject : AnyObject] {
        var ResultDict = [NSObject : AnyObject]()
        ResultDict = [NSObject : AnyObject](objects: dict, forKeys: (dict.valueForKey(key) as! String))
        return ResultDict
    }
}